function output=datreader(path, sweeps, flag)
%DATREADER imports GPatchM data files (.dat files) into Matlab
%DATREADER was written by Ernesto Vargas and Jeremy Treger in Francisco
%Bezanilla's lab
%OUTPUT = DATREADER(PATH, SWEEPS, FLAG)imports the traces listed in SWEEPS 
%from the .dat file located at PATH using the settings specified by FLAG.
%
%PATH is a string containging the filepath for the desired GPatchM .dat file.
%If the string 'd' is entered for PATH, a dialogue box will ask for the desired file. 
%
%SWEEPS is a row vector listing the sweep numbers to be imported. If the 
%string 'a' is entered for SWEEPS, the entire .dat file will be imported.
%
%The choice of FLAG determines whether to import traces as single sweeps or groups.
%   's':  Import the traces listed in SWEEPS as single traces. Separate
%         traces are stored as fields in a 1-D row structure.
%         
%   'g':  Import the whole groups containing each of the traces listed in SWEEPS.
%         Traces are stored in a 2-level structure. Each group is its own
%         substructure within the main structure called 'groupN' where N is 
%         the group number. Thus, if the main structure is called 'mydata'
%         then the second group of the file would be stored as
%         mydata.group2. 
%         Each of these substructures is itself a 1-D row structure
%         containing all the traces of the group. Thus the sixth trace of
%         the group discussed above would be accessed via mydata.group2(6).
%
%Examples:
%   data=datreader('d','a','g');
%       Imports an entire file, which will be speciifed in a popup dialogue
%       box, with the traces in groups
%
%       Once the data is imported, to access the fourth trace of the third
%       group of the file, type 'data.group3(4)'
%
%   sweep6=datreader('C:\Users\BC-Lab\Desktop\Data\myfile.dat',[6],'s');
%       Imports the 6th sweep of 'myfile' located at the stated path
%
%   file2_goodtraces=datreader('C:\Data\NewTest\myfile2.dat',[6,8:11,14],'s');
%       Imports sweeps 6,8,9,10,11,and 14 from 'myfile2' at the stated path as
%       single sweeps
%
%   groups=datreader('d',[25, 110],'g');
%       Imports the group containing trace 25 and the group containing trace 110
%       from a file which will be speciifed in a popup dialogue box


%--------------------------------------------------------------------------
% INPUT VARIABLE HANDLING
%--------------------------------------------------------------------------

%Get file path and ensure it conforms to a valid path
if strcmp(path,'d')
    [file, path]=uigetfile('*.dat');
    fid=fopen([path file],'r','l');
    assignin('base','filenameglobal',file);
else
    validateattributes(path,{'char'},{'row'},'','PATH',1);
    if length(path)<5 || ~strcmp(path(2:3),':\')
        disp('Error: PATH does not appear to be a valid file path')
        output=[];
        return
    end
    if strcmp(path(end-3:end),'.dat')
        fid=fopen(path,'r','l');
    else
        fid=fopen([path '.dat'],'r','l');
    end
end

%A subfunction is called to get relevant information about the data file;
%an error is thrown if the provided path is not a valid file
try
    [numpoints,groupindex,sweepindex,numgroups,numsweeps]=getprofile(fid);
catch %#ok<CTCH>
    disp('Error: PATH does not point to a valid .dat file')
    output=[];
    return
end
    
%Determine which sweeps to use and ensure input makes sense (a row vector of positive integers)
if strcmp(sweeps,'a')
    readentirefile=1;
else
    validateattributes(sweeps,{'numeric'},{'integer','positive','real','row'},'','SWEEPS',2);
    if max(sweeps)>numsweeps
        disp(['Error: Only ' int2str(numsweeps) ' sweeps in file'])
        output=[];
        return
    end
    readentirefile=0;
end

%Decide whether to import traces as sweeps or groups, display error if invalid flag provided
if strcmp(flag,'s')
    importasgroups=0;
elseif strcmp(flag,'g')
    importasgroups=1;
else
    disp('Error: FLAG must be ''s'' or ''g''')
    output=[];
    return
end

%--------------------------------------------------------------------------
%SUBFUNCTION CALLING
%--------------------------------------------------------------------------

if readentirefile==0
    if importasgroups==0
        %Import chosen traces as single sweeps
        for i=1:length(sweeps);
            currentsweep=sweeps(i);
            data(i)=readsinglesweep(fid,numpoints,currentsweep);
        end
        output=data;
    elseif importasgroups==1
        %Import groups containing chosen traces
        for i=1:length(sweeps);
            currentsweep=sweeps(i);
            data.(['group' num2str(i)])=readwholegroup(fid,numpoints,currentsweep,groupindex,sweepindex);
        end
        output=data;
    end
elseif readentirefile==1
    if importasgroups==0
        %Import entire file as single sweeps
        for i=1:numsweeps;
            data(i)=readsinglesweep(fid,numpoints,i);
        end
        output=data;
    elseif importasgroups==1
        %Import entire file as groups
        for i=1:numgroups
            ind=groupindex(i);
            firstsweep=find(sweepindex==ind);
            data.(['group' num2str(i)])=readwholegroup(fid,numpoints,firstsweep,groupindex,sweepindex);
        end
        output=data;
    end
end

fclose(fid);
end


%--------------------------------------------------------------------------
% SUBFUNCTIONS ARE BELOW - DO NOT MODIFY
%--------------------------------------------------------------------------

function [numpoints,groupindex,sweepindex,numgroups,numsweeps] = getprofile(fileid)
%Rewind file

fseek(fileid,0,'bof');

%Read entire file as string
string=char(fread(fileid,'*char'));
string=reshape(string,1,length(string));

%Find the place in the file where the sweeps/groups END
sweepindex=strfind(string,'FReco')+16;
groupindex=strfind(string,'FRecoE')+16;
%Find the total number of sweeps/groups
numsweeps=length(sweepindex);
numgroups=length(groupindex);
%Find the place in the file where the sweeps/groups START/END
sweepindex=[0,sweepindex];
groupindex=[0,groupindex];

tmp=sweepindex(2:end) - sweepindex(1:end-1);
%All sweeps must have the same number of points or else return error
if( length(unique(tmp)) ~= 1)
    error('Number of datapoints is not consistent in file');
end

% This array contains the number of datapoints per trace
numarr=(tmp-160)/2;
numpoints=numarr(1);

%Rewind file
fseek(fileid,0,'bof');
end

function data = readsinglesweep (fileid,numpoints,sweepnumber)
offset=(sweepnumber-1)*(numpoints*2+160);
msg=fseek(fileid,offset,'bof');
if msg
    error('Tried accessing beyond end of file');
end

dat	= fread(fileid, numpoints,'int16','l');
data.amp        = fread(fileid,  8, 'int16',  'l');
data.pf         = fread(fileid,  1, 'int16',  'l');
data.x2         = fread(fileid,  1, 'int16',  'l');
data.dur        = fread(fileid,  8, 'int32',  'l');
data.x3         = fread(fileid,  1, 'double', 'l');
data.durap 	    = fread(fileid,  1, 'int32',  'l');
data.durbp      = fread(fileid,  1, 'int32',  'l');
data.durbtwnsp  = fread(fileid,  1, 'int32',  'l');
data.durcp 	    = fread(fileid,  1, 'float32','l');
data.x6 	    = fread(fileid,  1, 'int16',  'l');
data.numsp 	    = fread(fileid,  1, 'int16',  'l');
data.numpulses  = fread(fileid,  1, 'int16',  'l');
data.trig1pulse = fread(fileid,  1, 'int16',  'l');
data.x7 	    = fread(fileid,  1, 'int16',  'l');
data.delay      = fread(fileid,  1, 'int16',  'l');
data.x8 	    = fread(fileid,  1, 'int16',  'l');
data.x9 	    = fread(fileid,  1, 'int16',  'l');
data.x0 	    = fread(fileid,  1, 'int16',  'l');
data.hp 	    = fread(fileid,  1, 'int16',  'l');
data.subhp      = fread(fileid,  1, 'int16',  'l');
data.y1 	    = fread(fileid,  1, 'int16',  'l');
data.y2 	    = fread(fileid,  1, 'int16',  'l');
data.acqfilter  = fread(fileid,  1, 'int16',  'l');
data.numavg	    = fread(fileid,  1, 'int16',  'l');
data.acqrate    = fread(fileid,  1, 'float32','l');
data.year 	    = fread(fileid,  1, 'int16',  'l');
data.month 	    = fread(fileid,  1, 'int8',   'l');
data.day 	    = fread(fileid,  1, 'int8',   'l');
data.hr 	    = fread(fileid,  1, 'int8',   'l');
data.min 	    = fread(fileid,  1, 'int8',   'l');
data.sec 	    = fread(fileid,  1, 'int8',   'l');
data.hsec 	    = fread(fileid,  1, 'int8',   'l');
data.y5 		= fread(fileid,  7, 'int16',  'l');
data.gain       = fread(fileid,  1, 'float32','l');
data.units 	    = fread(fileid,  1, 'float32','l');
data.y6 		= fread(fileid,  8, 'int8',   'l');
data.endgroup   = fread(fileid,  1, '*char',  'l');
data.y8 		= fread(fileid, 11, 'int8',   'l');

% Convert bits to mV and then to uA
mv = dat * (2^-15) * 10 * 1000;
uamp = (mv * data.units * 1e6 / data.gain);

data.data=uamp;
data.time=linspace(0,(numpoints-1)*data.acqrate,numpoints)';

data.sweep=sweepnumber;
end

function group=readwholegroup(fileid,numpoints,firstsweep,groupindex,sweepindex)
%Index of the sweep we are interested in
swpindex=(firstsweep-1)*(numpoints*2+160);
%Find group index smaller than index of desired sweep
groupindex<=swpindex;
%Which group index is the next smallest
gind=groupindex(groupindex<=swpindex);
gind=gind(end);
%What sweep does that correspond to?
currentsweep=find(sweepindex==gind);
%Read sweeps until end of group
i=1;
while 1
    dat=readsinglesweep(fileid,numpoints,currentsweep);
    currentsweep=currentsweep+1;
    group(i)=dat;
    if strcmp(dat.endgroup,'E')
        break
    end
    i=i+1;
end
end